/*
 * 
 * squid_mysql_auth: authentication via mysql for squid proxy server
 * 
 * Maintainer: Sterling Hughes <sterling@php.net>
 *
 * Author: Sterling Hughes
 * 
 * Usage:
 *   squid_mysql_auth [-d db] [-u username] [-p password] [-l port] [-s server] 
 *                    [-t table]
 * 
 * Dependencies: You need the mysql client libraries installed.
 * 
 * License: squid_mysql_auth is free software; you can redistribute it 
 * and/or modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2, 
 * or (at your option) any later version.
 *
 */

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

/* MySQL headers */
#include <mysql.h>

/* Versions of Mysql before 3.22 had a different syntax for
   the mysql_real_connect function, we test for that here */
#if defined(MYSQL_VERSION_ID) && MYSQL_VERSION_ID >= 32200
#define USE_NEW_REAL_CONNECT
#endif


#define CFG(__name) config->__name
#define SAFE_FREE(__var) if (__var) free(__var)


/* Define the global configuration struct, contains the user
   command line configuration variables */
typedef struct _mysql_auth_config
{
	char *host;
	char *database;
	char *table;
	char *db_username;
	char *db_password;
	char *login_field;
	char *password_field;

	unsigned int port;
}
mysql_auth_config;

static mysql_auth_config *config;


/* Arguments to this function, for getopt_long() */
static struct option arguments[] = {
    {"database",  required_argument, NULL, 'd'},
    {"username",  required_argument, NULL, 'u'},
    {"password",  required_argument, NULL, 'p'},
    {"port",      required_argument, NULL, 'l'},
    {"server",    required_argument, NULL, 's'},
    {"table",     required_argument, NULL, 't'},
	{"userfield", required_argument, NULL, 'l'},
	{"passfield", required_argument, NULL, 'a'}
};   

/* Allocate and initialize the configuration structure */
static int config_ctor()
{
	/* Allocate and zero out the config structure */
	config = (mysql_auth_config *) malloc(sizeof(mysql_auth_config));
	if (! config) {
		fprintf(stderr, "Couldn't allocate the MySQL configuration struct");
		return -1;
	}
	memset(config, 0, sizeof(mysql_auth_config));

	return 0;
}

/* Free the configuration structure */
static void config_dtor()
{
	SAFE_FREE( CFG(database) );
	SAFE_FREE( CFG(db_username) );
	SAFE_FREE( CFG(db_password) );
	SAFE_FREE( CFG(host) );
	SAFE_FREE( CFG(table) );
	SAFE_FREE( CFG(login_field) );
	SAFE_FREE( CFG(password_field) );

	free(config);
}


/* Parse the arguments and place them in the global configuration 
   structure, then set the defaults */
static void parse_arguments(int argc, char **argv)
{
	int opt_idx;
	int c = -1;

    do {
        c = getopt_long(argc, argv, "d:u:p:P:s:t:l:a", arguments, &opt_idx);
        switch (c) {
        case 'd':
            CFG(database) = strdup(optarg);
            break;
        case 'u':
            CFG(db_username) = strdup(optarg);
            break;
        case 'p':
            CFG(db_password) = strdup(optarg);
            break;
        case 'P':
            CFG(port) = (unsigned int) atoi(optarg);
            break;
        case 's':
            CFG(host) = strdup(optarg);
            break;
        case 't':
            CFG(table) = strdup(optarg);
            break;
		case 'l':
			CFG(login_field) = strdup(optarg);
			break;
		case 'a':
			CFG(password_field) = strdup(optarg);
			break;
        }
    } while (c != -1);

	/* Assign the defaults */
	if (! CFG(database)) {
		CFG(database) = strdup("squid");
	}

	if (! CFG(host)) {
		CFG(host) = strdup("localhost");
	}

	if (! CFG(table)) {
		CFG(table) = strdup("auth");
	}

	if (! CFG(login_field)) {
		CFG(login_field) = strdup("login");
	}

	if (! CFG(password_field)) {
		CFG(password_field) = strdup("password");
	}
}

/* Check the credentials given to us */
static int check_credentials(char *username, char *password)
{
	MYSQL      *conn;
	MYSQL_RES  *result;
	MYSQL_ROW   row;
	char       *query;
	char       *query_format;
	int         query_format_len = password ? 50 : 38;
	int         query_len;
	int         count;

	/* Allocate our query format */
	query_format = malloc(query_format_len + 1);
	if (password) 
		strcpy(query_format, 
		       "SELECT COUNT(*) FROM %s WHERE %s='%s' AND %s='%s'");
	else
		strcpy(query_format, "SELECT COUNT(*) FROM %s WHERE %s='%s'");

	/* Connect to the MySQL database */
	conn = mysql_init(NULL);
	if (conn == NULL) {
		fprintf(stderr, "Couldn't initialize the MySQL library");
		return -1;
	}

	if (
#ifdef USE_NEW_REAL_CONNECT
	mysql_real_connect(conn,
	                   CFG(host),
	                   CFG(db_username),
	                   CFG(db_password),
	                   CFG(database),
	                   CFG(port),
	                   NULL,
	                   0)
#else
	mysql_real_connect(conn,
	                   CFG(host),
	                   CFG(db_username),
	                   CFG(db_password),
	                   CFG(port),
	                   NULL,
	                   0)
#endif
	== NULL) {
		fprintf(stderr, 
		        "Couldn't establish connection to the mysql host %s on port %d",
		        CFG(host), CFG(port));
		return -1; 
	}

#ifndef USE_NEW_REAL_CONNECT
	if (mysql_select_db(conn, CFG(database)) != 0) {
		fprintf(stderr, "Couldn't select database %s",
		       CFG(database));
		mysql_close(conn);
		return -1;
	}
#endif

	/* Build the query */
	query_len = (query_format_len - 6)      + 
	            strlen(CFG(table))          +
	            strlen(CFG(login_field))    +
	            strlen(username);

	if (password) 
		query_len += (-4) + strlen(CFG(password_field)) + 
		             strlen(password);

	query = (char *) malloc(query_len + 1);
	if (! query) {
		fprintf(stderr, "Cannot allocate query string");
		return -1;
	}

	if (password)
		snprintf(query, query_len, query_format, 
		         CFG(table), CFG(login_field), username,
		         CFG(password_field), password);
	else
		snprintf(query, query_len, query_format,
		         CFG(table), CFG(login_field), username);

	/* Send the query */
	if (mysql_real_query(conn, query, query_len) != 0) {
		fprintf(stderr, "Could not send authentication query to the database server");
		return -1;
	}

	/* Fetch the result */
	result = mysql_store_result(conn);
	if (result == NULL) {
		fprintf(stderr, "Problem processing MySQL result set");
		return -1;
	}

	/* Fetch the row */
	row = mysql_fetch_row(result);
	if (row == NULL) {
		fprintf(stderr, "Couldn't fetch a row from the MySQL database");
		return -1;
	}
	count = atoi(row[0]);

	/* Cleanup */
	free(query_format);

	mysql_free_result(result);
	mysql_close(conn);

	/* If results are found count is greater than 0 therefore return 1
	   otherwise return 0 */
	return count ? 1 : 0;
}

#define BUFSIZE 256
#define VERIFIED "OK\n"
#define INVALID  "ERR\n"

int main(int argc, char **argv)
{
	char *username;
	char *password = NULL;
	char *p        = NULL;
	char  buf[BUFSIZE];

	/* Allocate and initialize the configuration structure */
	if (config_ctor() != 0) 
		exit(-1);

	/* Parse command line arguments */
	parse_arguments(argc, argv);

	/* Don't buffer output */
	setbuf(stdout, NULL);

	/* Loop through and grab data to process */
	while(fgets(buf, BUFSIZE, stdin) != NULL) {
		p = strchr(buf, '\n');
		if (p != NULL) *p = '\0';
		p = strchr(buf, '\r');
		if (p != NULL) *p = '\0';

		username = buf;
		p = strchr(buf, ' ');
		if (p != NULL) {
			password = p;
			*password++ = '\0';
		}

		if (check_credentials(username, password) > 0) {
			printf(VERIFIED);
		}
		else {
			printf(INVALID);
		}
	}

	/* Free configuration structure */
	config_dtor();

	return 0;
}
